#include "globals.h"
#include "game.h"
#include "physics.h"

static void MagneticHole(ball *b, int id, double hx, double hy, double r) {
  double dist=sqrt(sqr(b->px-hx)+sqr(b->py-hy));
  if(dist>r) return;
  if(dist<r-BallRadius) {
    b->falling=1;
    b->in_hole=id;
    if(IsAllowed(b->id,0))
      last_hole=id;
  } else {
    b->px=(1.0-HoleMagnetism*dt)*b->px+HoleMagnetism*dt*hx;
    b->py=(1.0-HoleMagnetism*dt)*b->py+HoleMagnetism*dt*hy;
  }
}


void UpdateBall(ball *b) {
  double speed,newspeed;
  bound *cb=&bounds[0];
  int i;

  if(!b->active) return;

  if(b->falling) {
    b->vz+=FallingSpeed*dt;
    b->pz+=b->vz*dt;
    if(b->pz>1.0) {
      b->vz=0.0;
      moving&=~b->bit;
      fallen|=b->bit;
      b->active=0;
      if(!preference && b->id && b->id!=8) {
        preference=b->id^(player<<3);
        first_collision=b->id;
      }
    }
    return;
  }

  if(b->vx==0.0 && b->vy==0.0 && b->vz==0.0) {
    moving&=~b->bit;
    return;
  }

  for(i=0; i<18; ++i, ++cb) {
    double sx=b->px-cb->px;
    double sy=b->py-cb->py;
    if((sx*cb->nx+sy*cb->ny)>-BallRadius){
      double lt=sx*cb->dx+sy*cb->dy;
      if(lt>=0.0 && lt<=cb->l) {
        double vn=b->vx*cb->nx+b->vy*cb->ny;
        if(vn>0.0) {
          double att=1.0-(BoundLoss*vn/sqrt(b->vx*b->vx+b->vy*b->vy));
          b->vx=att*(b->vx-2.0*vn*cb->nx);
          b->vy=att*(b->vy-2.0*vn*cb->ny);
          if(!b->id && (!(bound_mask&cb->bit))) {
            bound_mask|=cb->bit;
            ++bound_count;
          }
          break;
        }
      }
    }
  }

  MagneticHole(b,1, TableSizeX+EdgeExtend, TableSizeY+EdgeExtend, OuterHoleRadius);
  MagneticHole(b,2,                   0.0, TableSizeY+EdgeExtend, InnerHoleRadius);
  MagneticHole(b,3,-TableSizeX-EdgeExtend, TableSizeY+EdgeExtend, OuterHoleRadius);
  MagneticHole(b,4, TableSizeX+EdgeExtend,-TableSizeY-EdgeExtend, OuterHoleRadius);
  MagneticHole(b,5,                   0.0,-TableSizeY-EdgeExtend, InnerHoleRadius);
  MagneticHole(b,6,-TableSizeX-EdgeExtend,-TableSizeY-EdgeExtend, OuterHoleRadius);
  if(b->px<-TableSizeX || b->px>TableSizeX ||
     b->py<-TableSizeY || b->py>TableSizeY)
    b->falling=1;

  speed=sqrt(b->vx*b->vx+b->vy*b->vy);
  b->px+=b->vx*dt;
  b->py+=b->vy*dt;

  if(need_half_table && (!b->id) && b->px<=0.0 && (!first_collision))
    need_half_table=0;

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  glRotated(speed*dt/BallRadius*180.0/M_PI, b->vy, -b->vx, 0.0);
  glMultMatrixd(b->mat);
  glGetDoublev(GL_MODELVIEW_MATRIX,b->mat);
  glPopMatrix();

  newspeed=speed-Friction*dt;
  if(newspeed<0.0) {
    newspeed=0.0;
    moving&=~b->bit;
  } else {
    newspeed/=speed;
    moving|=b->bit;
  }
  b->vx=newspeed*b->vx;
  b->vy=newspeed*b->vy;
}


void Collide(ball *a, ball *b) {
  double dx,dy,d,comp,att;
  if(!(a->active && b->active)) return;
  if(a->falling || b->falling) return;
  dx=b->px-a->px;
  dy=b->py-a->py;
  d=dx*dx+dy*dy;
  if(d>4.0*BallRadius*BallRadius) return;
  if(!a->id && !first_collision) first_collision=b->id;
  comp=((a->vx-b->vx)*dx+(a->vy-b->vy)*dy)/d;
  if (comp<=0.0) return;
  att=1.0-CollisionLoss;
  a->vx=(a->vx-dx*comp)*att;
  a->vy=(a->vy-dy*comp)*att;
  b->vx=(b->vx+dx*comp)*att;
  b->vy=(b->vy+dy*comp)*att;
}
