#include "globals.h"
#include "physics.h"
#include "view.h"
#include "text.h"
#include "game.h"

static int initial_pos[15][2]={
      { 0,-2}, {-1,-1}, { 1,-1}, {-2, 0}, { 0, 0}, { 2, 0}, {-3, 1}, {-1, 1},
      { 1, 1}, { 3, 1}, {-4, 2}, {-2, 2}, { 0, 2}, { 2, 2}, { 4, 2}};

void InitGame(void) {
  int i,j,pmask=0;
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  for(i=0; i<=BallCount; ++i) {
    balls[i].id=i;
    balls[i].bit=1<<i;
    balls[i].active=1;
    balls[i].falling=0;
    balls[i].in_hole=0;
    glRotated(drand()*360.0,drand()-0.5,drand()-0.5,drand()-0.5);
    glGetDoublev(GL_MODELVIEW_MATRIX,balls[i].mat);
    if(i) {
      if(i==8) j=4; else
        do { j=rand()%15; } while(j==4 || (pmask&(1<<j)));
      pmask|=1<<j;
      balls[i].px=-0.5*TableSizeX-initial_pos[j][1]*BallRadius*1.74;
      balls[i].py=                initial_pos[j][0]*BallRadius*1.01;
    }
    balls[i].vx=0.0;
    balls[i].vy=0.0;
    balls[i].vz=0.0;
    balls[i].pz=0.0;
//if(!(balls[i].bit&0x8102)) balls[i].active=0;
  }
  balls[0].px=0.5*TableSizeX;
  balls[0].py=0.0;
  fouls[0]=0; fouls[1]=0;
  hole8[0]=0; hole8[1]=0;
  moving=0; player=0; preference=0; need_half_table=0;
  ResetCue(); //SetupWhiteBall();
}


static void CheckValidSetup(void) {
  double dist;
  int i;
  for(i=1; i<15; ++i)
    if(balls[i].active) {
      dist=sqr(balls[0].px-balls[i].px)+sqr(balls[0].py-balls[i].py);
      if(dist<4.0*BallRadius*BallRadius) {
        valid_setup=0;
        return;
      }
    }
  dist=sqr(     balls[0].px -TableSizeX-EdgeExtend)+
       sqr(fabs(balls[0].py)-TableSizeY-EdgeExtend);
  valid_setup=(dist>m_sqr(OuterHoleRadius+BallRadius));
}

void SetupWhiteBall(void) {
  balls[0].active=0;
  balls[0].falling=0;
  balls[0].px=0.5*TableSizeX;
  balls[0].py=0.0;
  balls[0].pz=0.0;
  balls[0].vx=0.0;
  balls[0].vy=0.0;
  balls[0].vz=0.0;
  if(!(rules&RULE_SETUP)) {
    ResetCue();
    return;
  }
  cue_angle=0.0;
  need_half_table=(rules&RULE_HALF_TABLE);
  CheckValidSetup();
  state=STATE_SETUP;
}


void FireCue(void) {
  double a;
  if(state!=STATE_CUE) return;
  a=cue_angle*M_PI/180.0;
  if(cue_power>MaxCuePower) cue_power=MaxCuePower;
  balls[0].vx=-cos(a)*cue_power;
  balls[0].vy=-sin(a)*cue_power;
  moving=balls[0].bit;
  fallen=0;
  first_collision=0; last_hole=0;
  bound_mask=0; bound_count=0;
  cue_power=0.0;
  cue_fired=t;
  state=STATE_ROLLING;
}


int MayPlayEightBall(void) {
  int i;
  if(rules&RULE_NO_PREFERENCE) return 1;
  if(!preference) return 0;
  if((preference^(player<<3))>8) {
    for(i=9; i<16; ++i) if(balls[i].active) return 0;
  } else {
    for(i=1; i<8; ++i) if(balls[i].active) return 0;
  }
  return 1;
}


int IsAllowed(int ball_num, int may_play_eight_ball) {
  if(rules&RULE_NO_PREFERENCE) return 1;
  if(!preference) return (ball_num!=8);
  if(ball_num==8 && may_play_eight_ball) return 1;
  if(!ball_num) return 0;
  if((preference^(player<<3))>8) return (ball_num>8);
                            else return (ball_num<8);
}


void ResetCue(void) {
  int i,nearest=0;
  int may_play_eight_ball=MayPlayEightBall();
  double nearest_dist=99999.9;
  for(i=1; i<16; ++i)
    if(balls[i].active && IsAllowed(i,may_play_eight_ball)) {
      double dist=sqr(balls[0].px-balls[i].px)+sqr(balls[0].py-balls[i].py);
      if(dist<nearest_dist) {
        nearest=i;
        nearest_dist=dist;
      }
    }
  if(nearest)
    cue_angle=atan2(balls[0].py-balls[nearest].py,
                      balls[0].px-balls[nearest].px)*180.0/M_PI;
  cue_cx=balls[0].px;
  cue_cy=balls[0].py;
  cue_power=0.0;
  balls[0].active=1;
  state=STATE_CUE;
}


int Foul(int reason_msg) {
  if(reason_msg>=0) SetMessage(reason_msg);
  if(reason_msg&MSG_END_GAME_BIT) {
    state=STATE_END;
    camera=CAMERA_AUTO;
    return 0;
  }
  if((rules&RULE_THREE_FOULS) && (++fouls[player]>=3))
    return Foul(MSG_LOSE_THREE_FOULS);
  player^=1;
  if((rules&RULE_SETUP_AFTER_FOUL) || (!balls[0].active))
    SetupWhiteBall();
  else {
    need_half_table=0;
    ResetCue();
  }
  return 0;
}

int EndOfRoll(void) {
  int i;
  int may_play_eight_ball=MayPlayEightBall();

  // activate 8-ball hole preference
  if((rules&RULE_OPPOSITE_HOLE) && may_play_eight_ball &&
     last_hole && (!hole8[player]))
    hole8[player]=7-last_hole;

  // winning
  if(rules&RULE_NO_PREFERENCE) {
    int bcount=0;
    for(i=1; i<16; ++i) if(balls[i].active) ++bcount;
    if(!bcount)
      return Foul(MSG_WIN);
  } else if(may_play_eight_ball && (!balls[8].active)) {
    if((rules&RULE_OPPOSITE_HOLE) && hole8[player] &&
       (balls[8].in_hole!=hole8[player]))
      return Foul(MSG_LOSE_INVALID_HOLE);
    return Foul(MSG_WIN);
  }

  // the evil 8-ball
  if(!(rules&RULE_NO_PREFERENCE))
    if((!balls[8].active) && (!may_play_eight_ball))
      return Foul(MSG_LOSE_EIGHTBALL);

  // white ball falling
  if(!balls[0].active)
    return Foul(MSG_FOUL_WHITE_FALLEN);

  // invalid ball fallen?
  if(fallen && (!(rules&RULE_NO_PREFERENCE)) && preference)
    for(i=1; i<16; ++i)
      if((fallen&balls[i].bit) && (!IsAllowed(i,may_play_eight_ball)))
        return Foul(MSG_FOUL_INVALID_FALLEN);

  // no collision / invalid collision
  if(rules&RULE_MUST_COLLIDE) {
    if(!first_collision && bound_count<3)
      return Foul(MSG_FOUL_NO_COLLISION);
    if(first_collision && (!IsAllowed(first_collision,may_play_eight_ball)))
      return Foul(MSG_FOUL_INVALID_COLLISION);
  }

  // half table
  if(need_half_table)
    return Foul(MSG_FOUL_HALF_TABLE);

  fouls[player]=0;
  if(!fallen) player^=1;
  need_half_table=0;
  ResetCue();
  return 0;
}

#if 0
int EndOfRoll(void) {
  int i;
  int allowed=preference^(player<<3);
  int acount=0,bcount=0;
  if(allowed<8) {
    for(i=1; i<8; ++i) if(!balls[i].active) ++acount;
  } else {
    for(i=9; i<16; ++i) if(!balls[i].active) ++acount;
  }
  // winning
  for(i=1; i<16; ++i) if(!balls[i].active) ++bcount;
  if(bcount>=15)
    return Foul(MSG_WIN);
  // 8-ball handling
  if((rules&RULE_EIGHT_BALL) && !balls[8].active && acount<7)
    return Foul(MSG_LOSE_EIGHTBALL);
  // white ball falling foul
  if(!balls[0].active)
    return Foul(MSG_FOUL_WHITE_FALLEN);
  // foul if there's no collision
  if((rules&RULE_MUST_COLLIDE) && !first_collision)
    return Foul(MSG_FOUL_NO_COLLISION);
  else if(preference) {
    // foul if an incorrect ball was touched first
    if((rules&RULE_PREFERRED_COLLIDE) &&
      ((allowed>8 && first_collision<8) || (allowed<8 && first_collision>8)))
      return Foul(MSG_FOUL_INVALID_COLLISION);
  }
  if(fallen) {
    // invalid ball fallen?
    if((rules&RULE_PREFERENCE) && preference) {
      if(allowed>8) {
        for(i=1; i<8; ++i) if(fallen&balls[i].bit)
                              return Foul(MSG_FOUL_INVALID_FALLEN);
      } else {
        for(i=9; i<16; ++i) if(fallen&balls[i].bit)
                              return Foul(MSG_FOUL_INVALID_FALLEN);
      }
    }
  } else
    player^=1;
  ResetCue();
  return 0;
}
#endif


#define CurrentPrecisionFactor ((keymap&KEY_SHIFT)?PrecisionFactor:1.0)

void ProcessSubFrame(void) {
  int i,j;
  double xdir,ydir;
  switch(state) {

    case STATE_ROLLING:
      for(i=0; i<15; ++i)
        for(j=i+1; j<16; ++j)
          Collide(&balls[i],&balls[j]);
      for(i=0; i<16; ++i)
        UpdateBall(&balls[i]);
      if(!moving) EndOfRoll();
      break;

    case STATE_CUE:
      if(keymap&KEY_LEFT)
        cue_angle-=dt*CurrentPrecisionFactor*CueMoveSpeed;
      if(keymap&KEY_RIGHT)
        cue_angle+=dt*CurrentPrecisionFactor*CueMoveSpeed;
      if(keymap&KEY_DOWN || keymap&KEY_SPACE)
        cue_power+=dt*CurrentPrecisionFactor*CuePowerSpeed;
      break;

    case STATE_SETUP:
      xdir=0.0; ydir=0.0;
      if(keymap&KEY_LEFT)  xdir= dt*CurrentPrecisionFactor*SetupMoveSpeed;
      if(keymap&KEY_RIGHT) xdir=-dt*CurrentPrecisionFactor*SetupMoveSpeed;
      if(keymap&KEY_UP)    ydir= dt*CurrentPrecisionFactor*SetupMoveSpeed;
      if(keymap&KEY_DOWN)  ydir=-dt*CurrentPrecisionFactor*SetupMoveSpeed;
      if(camera==CAMERA_NEAR || camera==CAMERA_FAR) {
        balls[0].px-=ydir;
        balls[0].py+=xdir;
      } else {
        balls[0].px+=xdir;
        balls[0].py+=ydir;
      }
      if(balls[0].px>TableSizeX-BallRadius)   balls[0].px=TableSizeX-BallRadius;
      if(balls[0].px<TableSizeX*0.5)          balls[0].px=TableSizeX*0.5;
      if(balls[0].py>TableSizeY-BallRadius)   balls[0].py=TableSizeY-BallRadius;
      if(balls[0].py<BallRadius-TableSizeY)   balls[0].py=BallRadius-TableSizeY;
      CheckValidSetup();
      break;

  }

  if(keymap&(KEY_PGUP|KEY_PGDN|KEY_PLUS|KEY_MINUS)) {
    int lcam=GetLogicalCamera();
    if(lcam<CONFIGURABLE_CAMERAS) {
      if(keymap&KEY_PGUP)
        if((CamHeight[lcam]+=dt*CurrentPrecisionFactor*CamHeightSpeed)>MaxCamHeight)
          CamHeight[lcam]=MaxCamHeight;
      if(keymap&KEY_PGDN)
        if((CamHeight[lcam]-=dt*CurrentPrecisionFactor*CamHeightSpeed)<0.0)
          CamHeight[lcam]=0.0;
      if(keymap&KEY_PLUS)
        if((CamDist[lcam]-=dt*CurrentPrecisionFactor*CamDistSpeed)<MinCamDist)
          CamDist[lcam]=MinCamDist;
      if(keymap&KEY_MINUS)
        if((CamDist[lcam]+=dt*CurrentPrecisionFactor*CamDistSpeed)>MaxCamDist)
          CamDist[lcam]=MaxCamDist;
    }
  }

  UpdateBox(&BHelp);
  UpdateBox(&BRules);
  SetView(); UpdateView();
}
